<?php

require_once __DIR__ . '/../vendor/autoload.php';

use Core\Router;
use Core\Session;

// Start session
Session::start();

// Load configuration
$config = require_once __DIR__ . '/../config/app.php';

// Error reporting based on environment
if ($config['debug']) {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
}

// Set timezone
date_default_timezone_set($config['timezone']);

// Initialize router
$router = new Router();

// Authentication Routes
$router->get('/', [App\Controllers\DashboardController::class, 'index'], 'dashboard');
$router->get('/login', [App\Controllers\AuthController::class, 'showLogin'], 'login');
$router->post('/login', [App\Controllers\AuthController::class, 'login']);
$router->get('/register', [App\Controllers\AuthController::class, 'showRegister'], 'register');
$router->post('/register', [App\Controllers\AuthController::class, 'register']);
$router->get('/forgot-password', [App\Controllers\AuthController::class, 'showForgotPassword'], 'forgot-password');
$router->post('/forgot-password', [App\Controllers\AuthController::class, 'forgotPassword']);
$router->get('/reset-password/{token}', [App\Controllers\AuthController::class, 'showResetPassword'], 'reset-password');
$router->post('/reset-password', [App\Controllers\AuthController::class, 'resetPassword']);
$router->get('/logout', [App\Controllers\AuthController::class, 'logout'], 'logout');

// Dashboard Routes
$router->get('/dashboard', [App\Controllers\DashboardController::class, 'index'], 'dashboard.index');

// Contact Routes
$router->get('/contacts', [App\Controllers\ContactController::class, 'index'], 'contacts.index');
$router->get('/contacts/create', [App\Controllers\ContactController::class, 'create'], 'contacts.create');
$router->post('/contacts', [App\Controllers\ContactController::class, 'store'], 'contacts.store');
$router->get('/contacts/{id}', [App\Controllers\ContactController::class, 'show'], 'contacts.show');
$router->get('/contacts/{id}/edit', [App\Controllers\ContactController::class, 'edit'], 'contacts.edit');
$router->post('/contacts/{id}', [App\Controllers\ContactController::class, 'update'], 'contacts.update');
$router->delete('/contacts/{id}', [App\Controllers\ContactController::class, 'destroy'], 'contacts.destroy');
$router->get('/contacts/import', [App\Controllers\ContactController::class, 'showImport'], 'contacts.import');
$router->post('/contacts/import', [App\Controllers\ContactController::class, 'import']);
$router->get('/contacts/export', [App\Controllers\ContactController::class, 'export'], 'contacts.export');
$router->post('/contacts/extract-group', [App\Controllers\ContactController::class, 'extractGroup'], 'contacts.extract-group');

// Group Routes
$router->get('/groups', [App\Controllers\ContactController::class, 'groups'], 'groups.index');
$router->post('/groups', [App\Controllers\ContactController::class, 'createGroup'], 'groups.store');
$router->get('/groups/{id}', [App\Controllers\ContactController::class, 'showGroup'], 'groups.show');
$router->post('/groups/{id}', [App\Controllers\ContactController::class, 'updateGroup'], 'groups.update');
$router->delete('/groups/{id}', [App\Controllers\ContactController::class, 'deleteGroup'], 'groups.destroy');

// Template Routes
$router->get('/templates', [App\Controllers\TemplateController::class, 'index'], 'templates.index');
$router->get('/templates/create', [App\Controllers\TemplateController::class, 'create'], 'templates.create');
$router->post('/templates', [App\Controllers\TemplateController::class, 'store'], 'templates.store');
$router->get('/templates/{id}', [App\Controllers\TemplateController::class, 'show'], 'templates.show');
$router->get('/templates/{id}/edit', [App\Controllers\TemplateController::class, 'edit'], 'templates.edit');
$router->post('/templates/{id}', [App\Controllers\TemplateController::class, 'update'], 'templates.update');
$router->delete('/templates/{id}', [App\Controllers\TemplateController::class, 'destroy'], 'templates.destroy');
$router->post('/templates/{id}/submit', [App\Controllers\TemplateController::class, 'submit'], 'templates.submit');

// Campaign Routes
$router->get('/campaigns', [App\Controllers\CampaignController::class, 'index'], 'campaigns.index');
$router->get('/campaigns/create', [App\Controllers\CampaignController::class, 'create'], 'campaigns.create');
$router->post('/campaigns', [App\Controllers\CampaignController::class, 'store'], 'campaigns.store');
$router->get('/campaigns/{id}', [App\Controllers\CampaignController::class, 'show'], 'campaigns.show');
$router->get('/campaigns/{id}/edit', [App\Controllers\CampaignController::class, 'edit'], 'campaigns.edit');
$router->post('/campaigns/{id}', [App\Controllers\CampaignController::class, 'update'], 'campaigns.update');
$router->delete('/campaigns/{id}', [App\Controllers\CampaignController::class, 'destroy'], 'campaigns.destroy');
$router->post('/campaigns/{id}/start', [App\Controllers\CampaignController::class, 'start'], 'campaigns.start');
$router->post('/campaigns/{id}/pause', [App\Controllers\CampaignController::class, 'pause'], 'campaigns.pause');
$router->post('/campaigns/{id}/cancel', [App\Controllers\CampaignController::class, 'cancel'], 'campaigns.cancel');

// Inbox Routes
$router->get('/inbox', [App\Controllers\InboxController::class, 'index'], 'inbox.index');
$router->get('/inbox/{id}', [App\Controllers\InboxController::class, 'show'], 'inbox.show');
$router->post('/inbox/{id}/reply', [App\Controllers\InboxController::class, 'reply'], 'inbox.reply');
$router->post('/inbox/{id}/close', [App\Controllers\InboxController::class, 'close'], 'inbox.close');

// Analytics Routes
$router->get('/analytics', [App\Controllers\AnalyticsController::class, 'index'], 'analytics.index');
$router->get('/analytics/campaigns', [App\Controllers\AnalyticsController::class, 'campaigns'], 'analytics.campaigns');
$router->get('/analytics/export', [App\Controllers\AnalyticsController::class, 'export'], 'analytics.export');

// Settings Routes
$router->get('/settings', [App\Controllers\SettingsController::class, 'index'], 'settings.index');
$router->get('/settings/account', [App\Controllers\SettingsController::class, 'account'], 'settings.account');
$router->post('/settings/account', [App\Controllers\SettingsController::class, 'updateAccount']);
$router->get('/settings/billing', [App\Controllers\SettingsController::class, 'billing'], 'settings.billing');
$router->get('/settings/integrations', [App\Controllers\SettingsController::class, 'integrations'], 'settings.integrations');
$router->post('/settings/integrations', [App\Controllers\SettingsController::class, 'updateIntegrations']);

// Billing Routes
$router->get('/billing/plans', [App\Controllers\BillingController::class, 'plans'], 'billing.plans');
$router->post('/billing/subscribe', [App\Controllers\BillingController::class, 'subscribe'], 'billing.subscribe');
$router->get('/billing/payments', [App\Controllers\BillingController::class, 'payments'], 'billing.payments');
$router->post('/billing/payment/verify', [App\Controllers\BillingController::class, 'verifyPayment'], 'billing.verify');
$router->post('/billing/cancel', [App\Controllers\BillingController::class, 'cancel'], 'billing.cancel');

// Admin Routes
$router->get('/admin', [App\Controllers\AdminController::class, 'index'], 'admin.index');
$router->get('/admin/users', [App\Controllers\AdminController::class, 'users'], 'admin.users');
$router->post('/admin/users/{id}/status', [App\Controllers\AdminController::class, 'updateUserStatus']);
$router->get('/admin/settings', [App\Controllers\AdminController::class, 'settings'], 'admin.settings');
$router->post('/admin/settings', [App\Controllers\AdminController::class, 'updateSettings']);
$router->get('/admin/payments', [App\Controllers\AdminController::class, 'payments'], 'admin.payments');
$router->post('/admin/payments/{id}/approve', [App\Controllers\AdminController::class, 'approvePayment']);

// Webhook Routes
$router->get('/webhook/whatsapp', [App\Controllers\WebhookController::class, 'verify'], 'webhook.verify');
$router->post('/webhook/whatsapp', [App\Controllers\WebhookController::class, 'handle'], 'webhook.handle');

// API Routes (if enabled)
$router->post('/api/messages/send', [App\Controllers\ApiController::class, 'sendMessage']);
$router->get('/api/contacts', [App\Controllers\ApiController::class, 'getContacts']);
$router->post('/api/contacts', [App\Controllers\ApiController::class, 'createContact']);

// Dispatch the request
try {
    $router->dispatch();
} catch (Exception $e) {
    if ($config['debug']) {
        echo '<pre>';
        echo 'Error: ' . $e->getMessage() . "\n";
        echo 'File: ' . $e->getFile() . ':' . $e->getLine() . "\n";
        echo 'Trace: ' . $e->getTraceAsString();
        echo '</pre>';
    } else {
        http_response_code(500);
        echo '500 - Internal Server Error';
    }
}
