<?php

namespace Core;

class Router
{
    private $routes = [];
    private $namedRoutes = [];
    
    public function get($path, $callback, $name = null)
    {
        $this->addRoute('GET', $path, $callback, $name);
    }
    
    public function post($path, $callback, $name = null)
    {
        $this->addRoute('POST', $path, $callback, $name);
    }
    
    public function put($path, $callback, $name = null)
    {
        $this->addRoute('PUT', $path, $callback, $name);
    }
    
    public function delete($path, $callback, $name = null)
    {
        $this->addRoute('DELETE', $path, $callback, $name);
    }
    
    private function addRoute($method, $path, $callback, $name = null)
    {
        $pattern = $this->convertToRegex($path);
        $this->routes[$method][$pattern] = [
            'callback' => $callback,
            'path' => $path
        ];
        
        if ($name) {
            $this->namedRoutes[$name] = $path;
        }
    }
    
    private function convertToRegex($path)
    {
        // Convert route parameters like {id} to regex
        $pattern = preg_replace('/\{([a-zA-Z0-9_]+)\}/', '(?P<$1>[^/]+)', $path);
        return '#^' . $pattern . '$#';
    }
    
    public function dispatch()
    {
        $method = $_SERVER['REQUEST_METHOD'];
        $uri = $this->getUri();
        
        if (!isset($this->routes[$method])) {
            $this->notFound();
            return;
        }
        
        foreach ($this->routes[$method] as $pattern => $route) {
            if (preg_match($pattern, $uri, $matches)) {
                $params = array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);
                return $this->executeCallback($route['callback'], $params);
            }
        }
        
        $this->notFound();
    }
    
    private function executeCallback($callback, $params = [])
    {
        if (is_array($callback)) {
            [$controller, $method] = $callback;
            
            if (!class_exists($controller)) {
                throw new \Exception("Controller {$controller} not found");
            }
            
            $controllerInstance = new $controller();
            
            if (!method_exists($controllerInstance, $method)) {
                throw new \Exception("Method {$method} not found in {$controller}");
            }
            
            return call_user_func_array([$controllerInstance, $method], $params);
        }
        
        if (is_callable($callback)) {
            return call_user_func_array($callback, $params);
        }
        
        throw new \Exception("Invalid callback");
    }
    
    private function getUri()
    {
        $uri = $_SERVER['REQUEST_URI'];
        
        // Remove query string
        if (($pos = strpos($uri, '?')) !== false) {
            $uri = substr($uri, 0, $pos);
        }
        
        // Remove base path if in subdirectory
        $scriptName = dirname($_SERVER['SCRIPT_NAME']);
        if ($scriptName !== '/') {
            $uri = substr($uri, strlen($scriptName));
        }
        
        return '/' . trim($uri, '/');
    }
    
    private function notFound()
    {
        http_response_code(404);
        echo "404 - Page Not Found";
    }
    
    public function url($name, $params = [])
    {
        if (!isset($this->namedRoutes[$name])) {
            throw new \Exception("Route {$name} not found");
        }
        
        $path = $this->namedRoutes[$name];
        
        foreach ($params as $key => $value) {
            $path = str_replace('{' . $key . '}', $value, $path);
        }
        
        return $path;
    }
}
