<?php

namespace App\Models;

use Core\Model;

class User extends Model
{
    protected $table = 'users';
    protected $fillable = ['name', 'email', 'password', 'role', 'status'];
    
    public function findByEmail($email)
    {
        return $this->where(['email' => $email], null, 1);
    }
    
    public function create($data)
    {
        // Hash password before storing
        if (isset($data['password'])) {
            $data['password'] = password_hash($data['password'], PASSWORD_BCRYPT);
        }
        
        return parent::create($data);
    }
    
    public function verifyPassword($password, $hashedPassword)
    {
        return password_verify($password, $hashedPassword);
    }
    
    public function getSubscription($userId)
    {
        $sql = "SELECT * FROM subscriptions WHERE user_id = :user_id ORDER BY created_at DESC LIMIT 1";
        return $this->fetch($sql, ['user_id' => $userId]);
    }
    
    public function hasActiveSubscription($userId)
    {
        $subscription = $this->getSubscription($userId);
        
        if (!$subscription) {
            return false;
        }
        
        return $subscription['status'] === 'active' || 
               ($subscription['status'] === 'trial' && strtotime($subscription['trial_ends_at']) > time());
    }
    
    public function canSendMessages($userId)
    {
        $subscription = $this->getSubscription($userId);
        
        if (!$subscription) {
            return false;
        }
        
        if (!$this->hasActiveSubscription($userId)) {
            return false;
        }
        
        // Check if user has exceeded message limit
        if ($subscription['messages_limit'] !== -1) {
            return $subscription['messages_used'] < $subscription['messages_limit'];
        }
        
        return true;
    }
    
    public function incrementMessageCount($userId)
    {
        $sql = "UPDATE subscriptions 
                SET messages_used = messages_used + 1 
                WHERE user_id = :user_id 
                ORDER BY created_at DESC 
                LIMIT 1";
        
        return $this->query($sql, ['user_id' => $userId]);
    }
    
    public function createPasswordReset($email, $token)
    {
        $expiresAt = date('Y-m-d H:i:s', strtotime('+1 hour'));
        
        return $this->db->insert('password_resets', [
            'email' => $email,
            'token' => $token,
            'expires_at' => $expiresAt
        ]);
    }
    
    public function findPasswordReset($token)
    {
        $sql = "SELECT * FROM password_resets 
                WHERE token = :token 
                AND expires_at > NOW() 
                ORDER BY created_at DESC 
                LIMIT 1";
        
        return $this->fetch($sql, ['token' => $token]);
    }
    
    public function deletePasswordReset($token)
    {
        return $this->db->delete('password_resets', 'token = :token', ['token' => $token]);
    }
}
