<?php

namespace App\Controllers;

use Core\Controller;
use Core\Session;
use App\Models\User;

class AuthController extends Controller
{
    private $userModel;
    
    public function __construct()
    {
        parent::__construct();
        $this->userModel = new User();
    }
    
    public function showLogin()
    {
        // Redirect if already logged in
        if (Session::has('user_id')) {
            return $this->redirect('/dashboard');
        }
        
        return $this->render('auth.login');
    }
    
    public function login()
    {
        $email = $this->getInput('email');
        $password = $this->getInput('password');
        $remember = $this->getInput('remember');
        
        // Validation
        $errors = $this->validate($_POST, [
            'email' => 'required|email',
            'password' => 'required'
        ]);
        
        if (!empty($errors)) {
            Session::set('errors', $errors);
            Session::setOldInput($_POST);
            return $this->redirect('/login');
        }
        
        // Find user
        $user = $this->userModel->findByEmail($email);
        
        if (!$user || !$this->userModel->verifyPassword($password, $user['password'])) {
            Session::flash('error', 'Invalid email or password');
            Session::setOldInput($_POST);
            return $this->redirect('/login');
        }
        
        // Check if user is active
        if ($user['status'] !== 'active') {
            Session::flash('error', 'Your account has been suspended. Please contact support.');
            return $this->redirect('/login');
        }
        
        // Set session
        Session::set('user_id', $user['id']);
        Session::set('user_name', $user['name']);
        Session::set('user_email', $user['email']);
        Session::set('user_role', $user['role']);
        
        // Regenerate session ID for security
        Session::regenerate();
        
        // Handle remember me
        if ($remember) {
            $token = bin2hex(random_bytes(32));
            $this->userModel->update($user['id'], ['remember_token' => $token]);
            setcookie('remember_token', $token, time() + (86400 * 30), '/'); // 30 days
        }
        
        Session::flash('success', 'Welcome back, ' . $user['name'] . '!');
        return $this->redirect('/dashboard');
    }
    
    public function showRegister()
    {
        if (Session::has('user_id')) {
            return $this->redirect('/dashboard');
        }
        
        $config = require __DIR__ . '/../../config/payment.php';
        $plans = $config['plans'];
        
        return $this->render('auth.register', ['plans' => $plans]);
    }
    
    public function register()
    {
        $name = $this->getInput('name');
        $email = $this->getInput('email');
        $password = $this->getInput('password');
        $passwordConfirm = $this->getInput('password_confirm');
        $plan = $this->getInput('plan', 'starter');
        
        // Validation
        $errors = $this->validate($_POST, [
            'name' => 'required|min:3',
            'email' => 'required|email|unique:users',
            'password' => 'required|min:8'
        ]);
        
        if ($password !== $passwordConfirm) {
            $errors['password_confirm'] = ['Passwords do not match'];
        }
        
        if (!empty($errors)) {
            Session::set('errors', $errors);
            Session::setOldInput($_POST);
            return $this->redirect('/register');
        }
        
        try {
            $this->db->getConnection()->beginTransaction();
            
            // Create user
            $userId = $this->userModel->create([
                'name' => $name,
                'email' => $email,
                'password' => $password,
                'role' => 'user',
                'status' => 'active'
            ]);
            
            // Create trial subscription
            $config = require __DIR__ . '/../../config/payment.php';
            $planConfig = $config['plans'][$plan];
            $trialDays = $config['trial']['duration_days'];
            
            $subscriptionData = [
                'user_id' => $userId,
                'plan' => $plan,
                'billing_cycle' => 'monthly',
                'status' => 'trial',
                'trial_ends_at' => date('Y-m-d H:i:s', strtotime("+{$trialDays} days")),
                'current_period_start' => date('Y-m-d H:i:s'),
                'current_period_end' => date('Y-m-d H:i:s', strtotime('+1 month')),
                'messages_used' => 0,
                'messages_limit' => $planConfig['features']['messages_per_month']
            ];
            
            $this->db->insert('subscriptions', $subscriptionData);
            
            $this->db->getConnection()->commit();
            
            // Auto-login
            Session::set('user_id', $userId);
            Session::set('user_name', $name);
            Session::set('user_email', $email);
            Session::set('user_role', 'user');
            Session::regenerate();
            
            Session::flash('success', 'Account created successfully! Your trial period has started.');
            return $this->redirect('/dashboard');
            
        } catch (\Exception $e) {
            $this->db->getConnection()->rollback();
            Session::flash('error', 'Failed to create account. Please try again.');
            Session::setOldInput($_POST);
            return $this->redirect('/register');
        }
    }
    
    public function showForgotPassword()
    {
        return $this->render('auth.forgot-password');
    }
    
    public function forgotPassword()
    {
        $email = $this->getInput('email');
        
        $errors = $this->validate($_POST, [
            'email' => 'required|email'
        ]);
        
        if (!empty($errors)) {
            Session::set('errors', $errors);
            return $this->redirect('/forgot-password');
        }
        
        $user = $this->userModel->findByEmail($email);
        
        if ($user) {
            $token = bin2hex(random_bytes(32));
            $this->userModel->createPasswordReset($email, $token);
            
            // In production, send email here
            // For now, just flash the reset link
            $resetUrl = $_SERVER['HTTP_HOST'] . '/reset-password/' . $token;
            Session::flash('success', 'Password reset link: ' . $resetUrl);
        } else {
            // Don't reveal if email exists
            Session::flash('success', 'If an account exists with this email, a password reset link has been sent.');
        }
        
        return $this->redirect('/forgot-password');
    }
    
    public function showResetPassword($token)
    {
        $reset = $this->userModel->findPasswordReset($token);
        
        if (!$reset) {
            Session::flash('error', 'Invalid or expired reset token.');
            return $this->redirect('/login');
        }
        
        return $this->render('auth.reset-password', ['token' => $token]);
    }
    
    public function resetPassword()
    {
        $token = $this->getInput('token');
        $password = $this->getInput('password');
        $passwordConfirm = $this->getInput('password_confirm');
        
        $errors = $this->validate($_POST, [
            'password' => 'required|min:8'
        ]);
        
        if ($password !== $passwordConfirm) {
            $errors['password_confirm'] = ['Passwords do not match'];
        }
        
        if (!empty($errors)) {
            Session::set('errors', $errors);
            return $this->redirect('/reset-password/' . $token);
        }
        
        $reset = $this->userModel->findPasswordReset($token);
        
        if (!$reset) {
            Session::flash('error', 'Invalid or expired reset token.');
            return $this->redirect('/login');
        }
        
        $user = $this->userModel->findByEmail($reset['email']);
        
        if ($user) {
            $this->userModel->update($user['id'], ['password' => $password]);
            $this->userModel->deletePasswordReset($token);
            
            Session::flash('success', 'Password reset successfully. You can now login.');
            return $this->redirect('/login');
        }
        
        Session::flash('error', 'Failed to reset password.');
        return $this->redirect('/login');
    }
    
    public function logout()
    {
        Session::destroy();
        setcookie('remember_token', '', time() - 3600, '/');
        
        return $this->redirect('/login');
    }
}
